/**
 * Utility functions for handling recurring task periods
 */

/**
 * Get the current period date for a recurring task
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @param referenceDate - Optional date to use as reference (defaults to now)
 * @returns Date representing the start of the current period (time set to 00:00:00)
 */
export function getCurrentPeriodDate(
  recurrencePattern: string | null,
  referenceDate: Date = new Date()
): Date {
  if (!recurrencePattern) {
    throw new Error("Recurrence pattern is required");
  }

  const date = new Date(referenceDate);
  // Use UTC to avoid timezone issues
  date.setUTCHours(0, 0, 0, 0);

  if (recurrencePattern === "daily") {
    // Return today at midnight
    return date;
  } else if (recurrencePattern === "weekly") {
    // Return start of current week (Monday)
    const day = date.getUTCDay(); // 0 = Sunday, 1 = Monday, ..., 6 = Saturday
    const diff = day === 0 ? -6 : 1 - day; // Days to subtract to get to Monday
    const monday = new Date(date);
    monday.setUTCDate(date.getUTCDate() + diff);
    monday.setUTCHours(0, 0, 0, 0);
    monday.setUTCMinutes(0);
    monday.setUTCSeconds(0);
    monday.setUTCMilliseconds(0);
    return monday;
  } else if (recurrencePattern === "monthly") {
    // Return start of current month
    const firstDay = new Date(
      Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), 1)
    );
    firstDay.setUTCHours(0, 0, 0, 0);
    return firstDay;
  } else if (recurrencePattern === "semestral") {
    // Semestral: First semester (Jan-Jun) or Second semester (Jul-Dec)
    const month = date.getUTCMonth(); // 0-11
    const year = date.getUTCFullYear();
    // First semester: January 1 (month 0) to June 30
    // Second semester: July 1 (month 6) to December 31
    const semesterStartMonth = month < 6 ? 0 : 6; // January or July
    const firstDay = new Date(
      Date.UTC(year, semesterStartMonth, 1)
    );
    firstDay.setUTCHours(0, 0, 0, 0);
    return firstDay;
  }

  throw new Error(`Unknown recurrence pattern: ${recurrencePattern}`);
}

/**
 * Get the end date of the current period (deadline)
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @param referenceDate - Optional date to use as reference (defaults to now)
 * @returns Date representing the end of the current period (23:59:59)
 */
export function getPeriodEndDate(
  recurrencePattern: string | null,
  referenceDate: Date = new Date()
): Date {
  const startDate = getCurrentPeriodDate(recurrencePattern, referenceDate);
  const endDate = new Date(startDate);

  if (recurrencePattern === "daily") {
    // End of day (23:59:59)
    endDate.setHours(23, 59, 59, 999);
  } else if (recurrencePattern === "weekly") {
    // End of week (Sunday 23:59:59)
    endDate.setDate(endDate.getDate() + 6);
    endDate.setHours(23, 59, 59, 999);
  } else if (recurrencePattern === "monthly") {
    // End of month (last day 23:59:59)
    const lastDay = new Date(endDate.getFullYear(), endDate.getMonth() + 1, 0);
    lastDay.setHours(23, 59, 59, 999);
    return lastDay;
  } else if (recurrencePattern === "semestral") {
    // End of semester: June 30 or December 31
    const month = startDate.getUTCMonth();
    const year = startDate.getUTCFullYear();
    if (month === 0) {
      // First semester: ends June 30
      const lastDay = new Date(Date.UTC(year, 5, 30)); // June is month 5
      lastDay.setUTCHours(23, 59, 59, 999);
      return lastDay;
    } else {
      // Second semester: ends December 31
      const lastDay = new Date(Date.UTC(year, 11, 31)); // December is month 11
      lastDay.setUTCHours(23, 59, 59, 999);
      return lastDay;
    }
  }

  return endDate;
}

/**
 * Check if a given date is in the past for a recurring task
 * @param periodDate - The period date to check
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @param referenceDate - Optional date to use as reference (defaults to now)
 * @returns true if the period is in the past
 */
export function isPeriodInPast(
  periodDate: Date,
  recurrencePattern: string | null,
  referenceDate: Date = new Date()
): boolean {
  const currentPeriod = getCurrentPeriodDate(recurrencePattern, referenceDate);
  const checkDate = new Date(periodDate);
  checkDate.setHours(0, 0, 0, 0);

  return checkDate < currentPeriod;
}

/**
 * Get period date for a specific date (which period does this date belong to)
 * @param date - The date to check
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @returns Date representing the start of the period this date belongs to
 */
export function getPeriodDateForDate(
  date: Date,
  recurrencePattern: string | null
): Date {
  return getCurrentPeriodDate(recurrencePattern, date);
}

/**
 * Format period date for display
 * @param periodDate - The period date
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @returns Formatted string
 */
export function formatPeriodDate(
  periodDate: Date,
  recurrencePattern: string | null
): string {
  const date = new Date(periodDate);

  if (recurrencePattern === "daily") {
    return date.toLocaleDateString();
  } else if (recurrencePattern === "weekly") {
    const endDate = new Date(date);
    endDate.setDate(endDate.getDate() + 6);
    return `${date.toLocaleDateString()} - ${endDate.toLocaleDateString()}`;
  } else if (recurrencePattern === "monthly") {
    return date.toLocaleDateString("en-US", { month: "long", year: "numeric" });
  } else if (recurrencePattern === "semestral") {
    const month = date.getUTCMonth();
    const year = date.getUTCFullYear();
    if (month === 0) {
      // First semester: Jan - Jun
      return `First Semester ${year} (Jan - Jun)`;
    } else {
      // Second semester: Jul - Dec
      return `Second Semester ${year} (Jul - Dec)`;
    }
  }

  return date.toLocaleDateString();
}

/**
 * Get button text for completing a period
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @returns Button text
 */
export function getCompleteButtonText(
  recurrencePattern: string | null
): string {
  if (recurrencePattern === "daily") {
    return "Mark Today Complete";
  } else if (recurrencePattern === "weekly") {
    return "Mark This Week Complete";
  } else if (recurrencePattern === "monthly") {
    return "Mark This Month Complete";
  } else if (recurrencePattern === "semestral") {
    return "Mark This Semester Complete";
  }
  return "Mark Complete";
}

/**
 * Generate all periods for a recurring task from start date to end date
 * @param startDate - When the recurrence starts
 * @param endDate - When to stop generating periods (defaults to today + 30 days)
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @returns Array of period dates
 */
export function generateAllPeriods(
  startDate: Date,
  endDate: Date = new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // Default: 30 days from now
  recurrencePattern: string | null
): Date[] {
  if (!recurrencePattern) {
    return [];
  }

  const periods: Date[] = [];

  // Normalize start date to midnight (UTC to avoid timezone issues)
  const current = new Date(startDate);
  current.setUTCHours(0, 0, 0, 0);

  // Normalize end date to end of day (UTC)
  const end = new Date(endDate);
  end.setUTCHours(23, 59, 59, 999);

  // For daily tasks, we can generate periods more efficiently
  if (recurrencePattern === "daily") {
    let date = new Date(current);
    while (date <= end) {
      const periodDate = new Date(date);
      periodDate.setUTCHours(0, 0, 0, 0);

      // Avoid duplicates
      if (
        periods.length === 0 ||
        periods[periods.length - 1].getTime() !== periodDate.getTime()
      ) {
        periods.push(new Date(periodDate));
      }

      date.setUTCDate(date.getUTCDate() + 1);
    }
  } else {
    // For weekly/monthly/semestral, use getCurrentPeriodDate
    while (current <= end) {
      const periodDate = getCurrentPeriodDate(recurrencePattern, current);

      // Ensure period date is normalized to midnight (UTC)
      const normalizedPeriodDate = new Date(periodDate);
      normalizedPeriodDate.setUTCHours(0, 0, 0, 0);

      // Avoid duplicates
      if (
        periods.length === 0 ||
        periods[periods.length - 1].getTime() !== normalizedPeriodDate.getTime()
      ) {
        periods.push(new Date(normalizedPeriodDate));
      }

      // Move to next period
      if (recurrencePattern === "weekly") {
        current.setDate(current.getDate() + 7);
      } else if (recurrencePattern === "monthly") {
        current.setMonth(current.getMonth() + 1);
      } else if (recurrencePattern === "semestral") {
        // Move to next semester (6 months)
        current.setMonth(current.getMonth() + 6);
      }
    }
  }

  return periods;
}

/**
 * Format period date with better formatting
 * @param periodDate - The period date
 * @param recurrencePattern - "daily" | "weekly" | "monthly" | "semestral"
 * @returns Formatted string with day names
 */
export function formatPeriodDateDetailed(
  periodDate: Date,
  recurrencePattern: string | null
): string {
  const date = new Date(periodDate);

  if (recurrencePattern === "daily") {
    return date.toLocaleDateString("en-US", {
      weekday: "long",
      year: "numeric",
      month: "long",
      day: "numeric",
    });
  } else if (recurrencePattern === "weekly") {
    const endDate = new Date(date);
    endDate.setDate(endDate.getDate() + 6);
    return `${date.toLocaleDateString("en-US", {
      month: "short",
      day: "numeric",
    })} - ${endDate.toLocaleDateString("en-US", {
      month: "short",
      day: "numeric",
      year: "numeric",
    })}`;
  } else if (recurrencePattern === "monthly") {
    return date.toLocaleDateString("en-US", {
      month: "long",
      year: "numeric",
    });
  } else if (recurrencePattern === "semestral") {
    const month = date.getUTCMonth();
    const year = date.getUTCFullYear();
    if (month === 0) {
      // First semester: January 1 - June 30
      return `First Semester ${year} (January 1 - June 30)`;
    } else {
      // Second semester: July 1 - December 31
      return `Second Semester ${year} (July 1 - December 31)`;
    }
  }

  return date.toLocaleDateString();
}
